function ValidateAndPreprocessParams()
%% Check if input parameters are not conflicting.
%  Do preprocessing of the parameters, e.g.
%  1) if some bit matrices are used, then align num_e and num_i properly;
%  2) if some cells are watched, then switch from 1-based indexes used in Matlab to 0-based indexes used in C++;
%  and so on.

    global useGUI
    
    % Model parameters
    global num_e num_i
    global astro2PyraBinding
    
    % HPC parameters
    global useSPA saveInput2Output outFileName
    
    % RNG parameters
    global use32BitRng
    
    % Measured parameters
    global watchedCellIdx_e watchedCellIdx_i watchedCellNum_e watchedCellNum_i
    global watchedSynIdx_ee watchedSynIdx_ei watchedSynIdx_ie watchedSynIdx_ii
    global watchedSynNum_ee watchedSynNum_ei watchedSynNum_ie watchedSynNum_ii
    global enableAstro watchedAstroIdx watchedAstroNum
    global enableExtraCurrent_e watchedExtraCurrentIdx_e watchedExtraCurrentNum_e
    global enableExtraCurrent_i watchedExtraCurrentIdx_i watchedExtraCurrentNum_i
    global importMod_e watchedModCurrentIdx_e watchedModCurrentNum_e
    global importMod_i watchedModCurrentIdx_i watchedModCurrentNum_i
    
    %% Check if input parameters are not conflicting
    ValidateAndPreprocessParamsCore();
    
    %% Given types of synaptic conductance matrices,
    %  check if alignment for the number of neurons (num_e or num_i) is required.
    %  If so, do the alignment.
    AlignNumNeurons(true);

    %% Preprocess and validate the parameters specifying watched cells and synapses
    watchedCellNum_e = length(watchedCellIdx_e);
    watchedCellNum_i = length(watchedCellIdx_i);
    watchedAstroNum = length(watchedAstroIdx);
    watchedExtraCurrentNum_e = length(watchedExtraCurrentIdx_e);
    watchedExtraCurrentNum_i = length(watchedExtraCurrentIdx_i);
    watchedModCurrentNum_e = length(watchedModCurrentIdx_e);
    watchedModCurrentNum_i = length(watchedModCurrentIdx_i);
    watchedSynNum_ee = size(watchedSynIdx_ee, 1);
    watchedSynNum_ei = size(watchedSynIdx_ei, 1);
    watchedSynNum_ie = size(watchedSynIdx_ie, 1);
    watchedSynNum_ii = size(watchedSynIdx_ii, 1);
    watchedCellIdx_e = CheckAndPreprocWathedCellParams(watchedCellIdx_e, num_e, 'e-cell');
    watchedCellIdx_i = CheckAndPreprocWathedCellParams(watchedCellIdx_i, num_i, 'i-cell');
    if enableAstro
        num_a = GetNumAstrocytes(astro2PyraBinding, num_e, num_i);
        watchedAstroIdx = CheckAndPreprocWathedCellParams(watchedAstroIdx, num_a, 'astro cell');
    end
    watchedSynIdx_ee = CheckAndPreprocWathedSynParams(watchedSynIdx_ee, num_e, num_e, 'ee');
    watchedSynIdx_ei = CheckAndPreprocWathedSynParams(watchedSynIdx_ei, num_e, num_i, 'ei');
    watchedSynIdx_ie = CheckAndPreprocWathedSynParams(watchedSynIdx_ie, num_i, num_e, 'ie');
    watchedSynIdx_ii = CheckAndPreprocWathedSynParams(watchedSynIdx_ii, num_i, num_i, 'ii');
    if enableExtraCurrent_e
        watchedExtraCurrentIdx_e = CheckAndPreprocWathedCellParams(watchedExtraCurrentIdx_e, num_e, 'e-cell');
    end
    if enableExtraCurrent_i
        watchedExtraCurrentIdx_i = CheckAndPreprocWathedCellParams(watchedExtraCurrentIdx_i, num_i, 'i-cell');
    end
    if importMod_e
        watchedModCurrentIdx_e = CheckAndPreprocWathedCellParams(watchedModCurrentIdx_e, num_e, 'e-cell');
    end
    if importMod_i
        watchedModCurrentIdx_i = CheckAndPreprocWathedCellParams(watchedModCurrentIdx_i, num_i, 'i-cell');
    end
    
    %% Validate parameters of random number generators
    assert(~(useSPA && ~use32BitRng), 'Cannot use fine-grained random number generator std::mt19937_64 with single precision arithmetics.');
    
    %% Preprocess output MAT-file name
    if ~endsWith(outFileName, '.mat', 'IgnoreCase', true)
        outFileName = [outFileName, '.mat'];
    end
    
    %% Preprocess parameters depending on useGUI
    if ~useGUI
        saveInput2Output = false;
    end
end

function watchedCellIdx = CheckAndPreprocWathedCellParams(watchedCellIdx, num, name)
%% Check and preprocess parameters of wathed cells

    if ~isempty(watchedCellIdx)
        
        pat1 = 'Some index of watched %s is out of range [1, %i].';
        pat2 = 'The index %i appears more than once in the array of indexes of watched %ss.';
    
        watchedCellIdx = sort(watchedCellIdx);
        
        assert(watchedCellIdx(1) >= 1 && watchedCellIdx(end) <= num, sprintf(pat1, name, num));
        
        for i = 1 : length(watchedCellIdx) - 1
            assert(watchedCellIdx(i) ~= watchedCellIdx(i + 1), sprintf(pat2, watchedCellIdx(i), name));
        end
        
        % Vector-columns are used in C++
        watchedCellIdx = watchedCellIdx';
    end
    
end

function watchedSynIdx = CheckAndPreprocWathedSynParams(watchedSynIdx, numRows, numCols, name)
%% Check and preprocess parameters of watched synapses

    if ~isempty(watchedSynIdx)
        
        pat1 = 'Some %s index for watched %s-synapses is out of range [1, %i].';
        pat2 = 'The %s-synapse with position {row = %i, column = %i} is specified more than once in the array of watched synapses.';
    
        % Firstly sort it by 2nd dimension (the watched synapse column),
        % then sort it by 1nd dimension (the watched synapse row)
        watchedSynIdx = sortrows(watchedSynIdx);
        
        % Make sure that all row indexes are in range
        assert(watchedSynIdx(1, 1) >= 1 && watchedSynIdx(end, 1) <= numRows, sprintf(pat1, 'row', name, numRows));
        
        for i = 1 : size(watchedSynIdx, 1)
            % Make sure that this column index is in range
            assert(watchedSynIdx(i, 2) >= 1 && watchedSynIdx(i, 2) <= numCols, sprintf(pat1, 'column', name, numCols));

            % Make sure that there are no duplicates
            if i ~= size(watchedSynIdx, 1)
                assert(any(watchedSynIdx(i, :) ~= watchedSynIdx(i + 1, :)), sprintf(pat2, name, watchedSynIdx(i, 1), watchedSynIdx(i, 2)));
            end
        end
    end
    
end