function GenerateInclFiles(modClassNames_e, modClassNames_i, outDirPath)
%% Generate all "*Incl.[h|cpp]" files

    % Generate "DistEnvIncl.h" file
    pat1 = '    extern MPI_Datatype MPI_%s_FLOAT;';
    pat2 = '    extern MPI_Datatype MPI_%s_DOUBLE;';
    GenerateInclFile('DistEnvIncl.h', @GenerateDistEnvIncl, modClassNames_e, modClassNames_i, outDirPath, pat1, pat2);
    
    % Generate "DistEnvIncl.cpp" file
    pat1 = '    MPI_Datatype MPI_%s_FLOAT;';
    pat2 = '    MPI_Datatype MPI_%s_DOUBLE;';
    GenerateInclFile('DistEnvIncl.cpp', @GenerateDistEnvIncl, modClassNames_e, modClassNames_i, outDirPath, pat1, pat2);

    % Generate "GetTypeTagUtilsIncl.h" file
    GenerateInclFile('GetTypeTagUtilsIncl.h', @GenerateGetTypeTagUtilsInclH, modClassNames_e, modClassNames_i, outDirPath);
    
    % Generate "LocalVectorIncl1.cpp" file
    GenerateInclFile('LocalVectorIncl1.cpp', @GenerateLocalVectorIncl1Cpp, modClassNames_e, modClassNames_i, outDirPath);
    
    % Generate "LocalVectorIncl2.cpp" file
    GenerateInclFile('LocalVectorIncl2.cpp', @GenerateVectorInclCpp, modClassNames_e, modClassNames_i, outDirPath, 'Local');
    
    % Generate "DistVectorIncl.cpp" file
    GenerateInclFile('DistVectorIncl.cpp', @GenerateVectorInclCpp, modClassNames_e, modClassNames_i, outDirPath, 'Dist');
    
    % Generate "MatFileReadUtilsIncl.cpp" file
    sig = 'DistVector<mod::T1<T2>> ReadCheckScatterSpecialVectorFromIntermediateMat<mod::T1<T2>>(const char *name, int length);';
    GenerateInclFile('MatFileReadUtilsIncl.cpp', @GenerateMatFileIOUtilsInclCpp, modClassNames_e, modClassNames_i, outDirPath, sig);
    
    % Generate "MatFileWriteUtilsIncl.cpp" file
    sig = 'void GatherWriteSpecialVectorToIntermediateMat<mod::T1<T2>>(const DistVector<mod::T1<T2>> &vector, const char *name);';
    GenerateInclFile('MatFileWriteUtilsIncl.cpp', @GenerateMatFileIOUtilsInclCpp, modClassNames_e, modClassNames_i, outDirPath, sig);
    
    % Generate "mainIncl1.cpp" file
    GenerateInclFile('mainIncl1.cpp', @GenerateMainIncl1Cpp, modClassNames_e, modClassNames_i, outDirPath);

    % Generate "mainIncl2.cpp" file
    GenerateInclFile('mainIncl2.cpp', @GenerateMainIncl2Cpp, modClassNames_e, modClassNames_i, outDirPath);

end

function GenerateInclFile(outFileName, Generator, modClassNames_e, modClassNames_i, outDirPath, varargin)
%% Generate one "*Incl.[h|cpp]" file

    fprintf('    Generating %s ...\n', outFileName);

    if isempty(modClassNames_e) && isempty(modClassNames_i)
        lines = {};
    else
        lines = Generator(modClassNames_e, modClassNames_i, varargin{:});
    end
    
    SaveLinesToFile(lines, outDirPath, outFileName);

end

function lines = GenerateDistEnvIncl(modClassNames_e, modClassNames_i, pat1, pat2)
%% Generate "DistEnvIncl.[h|cpp]" file

    lines = {'namespace DistEnv'; '{'};
    
    pat = '    // Declaring MPI types for MOD classes of %s-neurons';
    
    function Generate(modClassNames, neuronType)
    % Helper generator
        if isempty(modClassNames)
            return
        end
        lines{end + 1} = sprintf(pat, neuronType); %#ok<*AGROW>
        lines{end + 1} = '    ';
        for i = 1 : length(modClassNames)
            capModClassName = upper(modClassNames{i});
            lines{end + 1} = sprintf(pat1, capModClassName);    % float
            lines{end + 1} = sprintf(pat2, capModClassName);    % double
            lines{end + 1} = '    ';
        end
    end
    
    Generate(modClassNames_e, 'e');
    
    if ~isempty(modClassNames_e)
        lines{end + 1} = '    ';
    end
    
    Generate(modClassNames_i, 'i');
    
    lines{end} = '}';

end

function lines = GenerateGetTypeTagUtilsInclH(modClassNames_e, modClassNames_i)
%% Generate "GetTypeTagUtilsIncl.h" file

    % Generate #include statements for all the MOD classes
    lines = IncludeAllModClasses(modClassNames_e, modClassNames_i);
    
    pat1 = '// Getting MPI types of MOD classes for %s-neurons';
    pat2 = 'inline MPI_Datatype GetMpiDataType<mod::%s<float>>()';
    pat3 = '    return DistEnv::MPI_%s_FLOAT;';
    pat4 = 'inline MPI_Datatype GetMpiDataType<mod::%s<double>>()';
    pat5 = '    return DistEnv::MPI_%s_DOUBLE;';
    
    function Generate(modClassNames, neuronType)
    % Helper generator
        if isempty(modClassNames)
            return
        end
        lines{end + 1} = sprintf(pat1, neuronType);
        lines{end + 1} = '';
        for i = 1 : length(modClassNames)
            modClassName = modClassNames{i};
            capModClassName = upper(modClassName);
            % float
            lines{end + 1} = 'template <>';
            lines{end + 1} = sprintf(pat2, modClassName);
            lines{end + 1} = '{';
            lines{end + 1} = sprintf(pat3, capModClassName);
            lines{end + 1} = '}';
            lines{end + 1} = '';
            % double
            lines{end + 1} = 'template <>';
            lines{end + 1} = sprintf(pat4, modClassName);
            lines{end + 1} = '{';
            lines{end + 1} = sprintf(pat5, capModClassName);
            lines{end + 1} = '}';
            lines{end + 1} = '';
        end
    end
    
    Generate(modClassNames_e, 'e');
    
    if ~isempty(modClassNames_e)
        lines{end + 1} = '';
    end
    
    Generate(modClassNames_i, 'i');
    
end

function lines = GenerateLocalVectorIncl1Cpp(modClassNames_e, modClassNames_i)
%% Generate "LocalVectorIncl1.cpp" file

    % Generate #include statements for all the MOD classes
    lines = IncludeAllModClasses(modClassNames_e, modClassNames_i);
    
    pat1 = '// Instantiating method for MOD classes of %s-neurons';
    pat2 = 'LocalVector<uint8_t> LocalVector<mod::%s<float>>::CastReshape(uint8_t* typeMarker);';
    pat3 = 'LocalVector<mod::%s<float>> LocalVector<uint8_t>::CastReshape(mod::%s<float>* typeMarker);';
    pat4 = 'LocalVector<uint8_t> LocalVector<mod::%s<double>>::CastReshape(uint8_t* typeMarker);';
    pat5 = 'LocalVector<mod::%s<double>> LocalVector<uint8_t>::CastReshape(mod::%s<double>* typeMarker);';
    
    function Generate(modClassNames, neuronType)
    % Helper generator
        if isempty(modClassNames)
            return
        end
        lines{end + 1} = sprintf(pat1, neuronType);
        lines{end + 1} = '';
        for i = 1 : length(modClassNames)
            modClassName = modClassNames{i};
            % float
            %   MOD class to uint8_t
            lines{end + 1} = 'template';
            lines{end + 1} = sprintf(pat2, modClassName);
            lines{end + 1} = '';
            %   uint8_t to MOD class
            lines{end + 1} = 'template';
            lines{end + 1} = sprintf(pat3, modClassName, modClassName);
            lines{end + 1} = '';
            % double
            %   MOD class to uint8_t
            lines{end + 1} = 'template';
            lines{end + 1} = sprintf(pat4, modClassName);
            lines{end + 1} = '';
            %   uint8_t to MOD class
            lines{end + 1} = 'template';
            lines{end + 1} = sprintf(pat5, modClassName, modClassName);
            lines{end + 1} = '';
        end
    end
    
    Generate(modClassNames_e, 'e');
    
    if ~isempty(modClassNames_e)
        lines{end + 1} = '';
    end
    
    Generate(modClassNames_i, 'i');

end

function lines = GenerateVectorInclCpp(modClassNames_e, modClassNames_i, classNamePrefix)
%% Generate "[Local|Dist]VectorIncl.cpp" file

    % Generate #include statements for all the MOD classes
    lines = IncludeAllModClasses(modClassNames_e, modClassNames_i);
    
    pat1 = '// Instantiating %sVector for MOD classes of %s-neurons';
    pat2 = 'class %sVector<mod::%s<float>>;';
    pat3 = 'class %sVector<mod::%s<double>>;';
    
    function Generate(modClassNames, neuronType)
    % Helper generator
        if isempty(modClassNames)
            return
        end
        lines{end + 1} = sprintf(pat1, classNamePrefix, neuronType);
        lines{end + 1} = '';
        for i = 1 : length(modClassNames)
            modClassName = modClassNames{i};
            % float
            lines{end + 1} = 'template';
            lines{end + 1} = sprintf(pat2, classNamePrefix, modClassName);
            lines{end + 1} = '';
            % double
            lines{end + 1} = 'template';
            lines{end + 1} = sprintf(pat3, classNamePrefix, modClassName);
            lines{end + 1} = '';
        end
    end
    
    Generate(modClassNames_e, 'e');
    
    if ~isempty(modClassNames_e)
        lines{end + 1} = '';
    end
    
    Generate(modClassNames_i, 'i');

end

function lines = GenerateMatFileIOUtilsInclCpp(modClassNames_e, modClassNames_i, sig)
%% Generate "MatFile[Read|Write]UtilsIncl.cpp" file

    % Generate #include statements for all the MOD classes
    lines = IncludeAllModClasses(modClassNames_e, modClassNames_i);
    
    lines{end + 1} = '#define Instantiate(T1, T2) template \';
    lines{end + 1} = sig;
    
    lines{end + 1} = '';
    
    pat1 = '// Instantiating methods for %s-type neurons';
    pat2 = 'Instantiate(%s, %s)';
        
    function Generate(modClassNames, neuronType)
    % Helper generator
        if isempty(modClassNames)
            return
        end
        lines{end + 1} = sprintf(pat1, neuronType);
        lines{end + 1} = '';
        for i = 1 : length(modClassNames)
            modClassName = modClassNames{i};
            lines{end + 1} = sprintf(pat2, modClassName, 'float');
            lines{end + 1} = sprintf(pat2, modClassName, 'double');
            lines{end + 1} = '';
        end
    end

    Generate(modClassNames_e, 'e');
    
    if ~isempty(modClassNames_e)
        lines{end + 1} = '';
    end
    
    Generate(modClassNames_i, 'i');
    
    lines{end + 1} = '#undef Instantiate';

end

function lines = GenerateMainIncl1Cpp(modClassNames_e, modClassNames_i)
%% Generate "mainIncl1.cpp" file

    % Generate #include statements for all the MOD classes
    lines = IncludeAllModClasses(modClassNames_e, modClassNames_i);

end

function lines = GenerateMainIncl2Cpp(modClassNames_e, modClassNames_i)
%% Generate "mainIncl2.cpp" file

    lines = {};
    
    pat1 = '// Creating MPI types for MOD classes of %s-neurons';
    pat2 = 'MPI_Type_contiguous(sizeof(mod::%s<float>), MPI_BYTE, &MPI_%s_FLOAT);';
    pat3 = 'MPI_Type_commit(&MPI_%s_FLOAT);';
    pat4 = 'MPI_Type_contiguous(sizeof(mod::%s<double>), MPI_BYTE, &MPI_%s_DOUBLE);';
    pat5 = 'MPI_Type_commit(&MPI_%s_DOUBLE);';
    
    function Generate(modClassNames, neuronType)
    % Helper generator
        if isempty(modClassNames)
            return
        end
        lines{end + 1, 1} = sprintf(pat1, neuronType);
        lines{end + 1, 1} = '';
        for i = 1 : length(modClassNames)
            modClassName = modClassNames{i};
            capModClassName = upper(modClassName);
            % float
            lines{end + 1} = sprintf(pat2, modClassName, capModClassName);
            lines{end + 1} = sprintf(pat3, capModClassName);
            lines{end + 1} = '';
            % double
            lines{end + 1} = sprintf(pat4, modClassName, capModClassName);
            lines{end + 1} = sprintf(pat5, capModClassName);
            lines{end + 1} = '';
        end
    end
    
    Generate(modClassNames_e, 'e');
    
    if ~isempty(modClassNames_e)
        lines{end + 1} = '';
    end
    
    Generate(modClassNames_i, 'i');
    
end

function lines = IncludeAllModClasses(modClassNames_e, modClassNames_i)
%% Generate #include statements for all the MOD classes

    lines = {};
    
    pat = '#include "%s.h"';
    
    function Generate(modClassNames)
    % Helper generator
        for i = 1 : length(modClassNames)
            lines{end + 1, 1} = sprintf(pat, modClassNames{i});
        end
        if ~isempty(modClassNames)
            lines{end + 1} = '';
        end
    end
    
    Generate(modClassNames_e);
    
    Generate(modClassNames_i);
    
    lines{end + 1} = '';
    
end