#include "BasicCurrent.h"
#include "MatFileIO/MatFileIOUtils.h"


// Load data from input (and intermediate) MAT-file(s)
template <typename T>
void BasicCurrent<T>::ReadInputDataAllocateTemporaryArrays(int num)
{
    // The current itself
    I = DistVector<T>(num);

    bool enable = (basicCurrentFactor != 0);

    if (enable)
    {
        ReadCheckScatterInSavableVector<T>(n_local, n, "n_" + suffix, num);
        ReadCheckScatterInSavableVector<T>(h_local, h, "h_" + suffix, num);

        n_tmp = DistVector<T>(num);
        h_tmp = DistVector<T>(num);

        // Kinetic parameters

        h_v_1 = ReadCheckBroadcastScalar<T>("h_" + suffix + "_v_1");
        h_v_2 = ReadCheckBroadcastScalar<T>("h_" + suffix + "_v_2");
        h_a_1 = ReadCheckBroadcastScalar<T>("h_" + suffix + "_a_1");
        h_a_2 = ReadCheckBroadcastScalar<T>("h_" + suffix + "_a_2");
        h_b_1 = ReadCheckBroadcastScalar<T>("h_" + suffix + "_b_1");
        h_b_2 = ReadCheckBroadcastScalar<T>("h_" + suffix + "_b_2");

        m_v_1 = ReadCheckBroadcastScalar<T>("m_" + suffix + "_v_1");
        m_v_2 = ReadCheckBroadcastScalar<T>("m_" + suffix + "_v_2");
        m_a_1 = ReadCheckBroadcastScalar<T>("m_" + suffix + "_a_1");
        m_a_2 = ReadCheckBroadcastScalar<T>("m_" + suffix + "_a_2");
        m_b_1 = ReadCheckBroadcastScalar<T>("m_" + suffix + "_b_1");
        m_b_2 = ReadCheckBroadcastScalar<T>("m_" + suffix + "_b_2");

        n_v_1 = ReadCheckBroadcastScalar<T>("n_" + suffix + "_v_1");
        n_v_2 = ReadCheckBroadcastScalar<T>("n_" + suffix + "_v_2");
        n_a_1 = ReadCheckBroadcastScalar<T>("n_" + suffix + "_a_1");
        n_a_2 = ReadCheckBroadcastScalar<T>("n_" + suffix + "_a_2");
        n_b_1 = ReadCheckBroadcastScalar<T>("n_" + suffix + "_b_1");
        n_b_2 = ReadCheckBroadcastScalar<T>("n_" + suffix + "_b_2");

        c1 = ReadCheckBroadcastScalar<T>("c1_" + suffix);
        c2 = ReadCheckBroadcastScalar<T>("c2_" + suffix);
        c3 = ReadCheckBroadcastScalar<T>("c3_" + suffix);
        c4 = ReadCheckBroadcastScalar<T>("c4_" + suffix);

        switch (suffix[0])
        {
        case 'e':
            h_inf = &BasicCurrent<T>::h_e_inf;
            m_inf = &BasicCurrent<T>::m_e_inf;
            n_inf = &BasicCurrent<T>::n_e_inf;
            tau_h = &BasicCurrent<T>::tau_h_e;
            tau_n = &BasicCurrent<T>::tau_n_e;

            break;
        case 'i':
            // Extra kinetic parameter
            phi = ReadCheckBroadcastScalar<T>("phi");

            h_inf = &BasicCurrent<T>::h_i_inf;
            m_inf = &BasicCurrent<T>::m_i_inf;
            n_inf = &BasicCurrent<T>::n_i_inf;
            tau_h = &BasicCurrent<T>::tau_h_i;
            tau_n = &BasicCurrent<T>::tau_n_i;

            break;
        }
    }
    else
    {
        #pragma omp parallel
        {
            I.AssignZeros();
        }
    }
}


// Write data to intermediate MAT-file.
// The method is called on all ranks with any number of threads, but not inside "#pragma omp master" region.
template <typename T>
void BasicCurrent<T>::GatherWriteIntermediateData()
{
    using namespace DistEnv;

    // Gather distributed vectors into local ones
    n.Gather(n_local, true);
    h.Gather(h_local, true);

    // Write the vectors
    if (myRank == MASTER_RANK)
    {
        #pragma omp barrier
        #pragma omp master
        {
            WriteVector<T>(n_local, "n_" + suffix, MatFile::Intermediate);
            WriteVector<T>(h_local, "h_" + suffix, MatFile::Intermediate);
        }
        #pragma omp barrier
    }
}
