#pragma once

#include "Containers/LocalVector.h"


enum GTonicGABAModel
{
    First = 1,
    Second = 2
};

// Gamma-Aminobutyric Acid mechanisms (DeltaVGABA, GTonicGABA, VTonicGABA)
template <typename T>
class GABA
{
    
public:
    
    // GABA parameters
    T DeltaVGABA;
    T GTonicGABA;
    T VTonicGABA;
    T AlphaTonic;
    
    // Variables for dynamic tonic current conductance

    bool dynamicGTonicGABA, enableFreqDelay;
    int freqDelay;

    // The dynamic model type
    GTonicGABAModel gTonicGABAModel;

    // 1st model parameter
    T freq_i;

    // 2nd model parameter
    int num_spikes_i;

    // Constructors
    GABA();
    GABA(int num_i, int m_steps_prev, int m_steps, T dt);

    // I/O staff
    void ReadInputDataAllocateTemporaryArrays(bool continuationMode);
    void WriteOutputData(int num_steps);
    void GatherWriteIntermediateData();

    // GTonicGABA ODE integration
    void DoOneStepPart1();
    void DoOneStepPart2(int iter);

private:
    
    // Variables for dynamic tonic current conductance
    bool watchGTonicGABA;
    T GTonicGABA_old;
    T GABA_, GABA_old;
    LocalVector<T> GTonicGABA_vec;  // Container for watched data

    // GTonicGABA ODE integration

    // Member function pointer for DoOneStepPart1
    typedef void (GABA<T>::*MemberFuncType1)();
    MemberFuncType1 DoOneStepPart1_;

    // Member function pointer for DoOneStepPart2
    typedef void (GABA<T>::*MemberFuncType2)(int);
    MemberFuncType2 DoOneStepPart2_;

    // The member functions for 1st model
    void DoOneStepPart1_FirstModel();
    void DoOneStepPart2_FirstModel(int iter);

    // The member functions for 2nd model
    void DoOneStepPart1_SecondModel();
    void DoOneStepPart2_SecondModel(int iter);

    // General simulation parameters
    int num_i;
    int m_steps_prev, m_steps;
    T dt, dt05;

    // 1st model parameters
    T basicFrequency, GTonicGABAControl;
    T const1;   // num_i * Af

    // 2d model parameters
    T G0, Rate, GABAb;
    T const2;   // 1 / dt / V * N / Na

    // Common parameter of both models
    T Gpump;

    // ODEs right-hand parts

    // 1nd model
    inline T GTonic_rhp()
    {
        return const1 * (freq_i + basicFrequency) - Gpump * (GTonicGABA - GTonicGABAControl);
    }
    
    // 2nd model
    inline T GABA_rhp()
    {
        return const2 * num_spikes_i - Gpump * (GABA_ - GABAb);
    }

    // Auxiliary functions for 2nd model

    inline T GABAToGTonicForSecondModel(T GABA_)
    {
        return G0 + Rate * GABA_;
    }

    inline T GTonicToGABAForSecondModel(T GTonic)
    {
        return (GTonic - G0) / Rate;
    }
};