#pragma region Writers

template <typename T>
void WriteScalar(T scalar, const char *name, MatFile matFile)
{
    using namespace pMat;

    // Determine whether file "output.mat" or "intermediate.mat" or should be used to write to
    MATFile *pFile = nullptr;
    switch (matFile)
    {
    case MatFile::Output:
        {
            pFile = pOutMatFile;
            break;
        }
    case MatFile::Intermediate:
        {
            pFile = pIntermMatFile;
            break;
        }
    default:
        {
            cout << "\n>>>>> Bad matFile argument passed to WriteScalar method.\n" << flush;
            MPI_Abort(MPI_COMM_WORLD, -1);
        }
    }

    mxClassID classId = GetMxClassId<T>();

    // Each scalar in Matlab is matrix of size 1-by-1
    mxArray *pArray = mxCreateNumericMatrix(1, 1, classId, mxREAL);

    // Matlab requires the scalar to be stored in specially allocated memory
    T *pData = (T*)mxCalloc(1, sizeof(T));
    *pData = scalar;

    mxSetData(pArray, pData);

    int status = matPutVariable(pFile, name, pArray);
    if (status != 0)
    {
        printf("\n>>>>> Failed to put mxArray \"%s\" into MAT-file.\n", name);
        cout << flush;
        MPI_Abort(MPI_COMM_WORLD, -1);
    }

    // Free memory allocated for pArray AND memory allocated for pData
    mxDestroyArray(pArray);
}

template <typename T>
void WriteScalar(T scalar, const std::string &name, MatFile matFile)
{
    WriteScalar(scalar, name.c_str(), matFile);
}

template <typename T>
void WriteVector(const LocalVector<T> &vector, const char *name, MatFile matFile)
{
    using namespace pMat;

    // Determine whether file "output.mat" or "intermediate.mat" or should be used to write to
    MATFile *pFile = nullptr;
    switch (matFile)
    {
    case MatFile::Output:
        {
            pFile = pOutMatFile;
            break;
        }
    case MatFile::Intermediate:
        {
            pFile = pIntermMatFile;
            break;
        }
    default:
        {
            cout << "\n>>>>> Bad matFile argument passed to WriteVector method.\n" << flush;
            MPI_Abort(MPI_COMM_WORLD, -1);
        }
    }

    mxClassID classId = GetMxClassId<T>();

    mxArray *pArray = mxCreateNumericMatrix(vector.length, 1, classId, mxREAL);

    mxSetData(pArray, vector.data);

    int status = matPutVariable(pFile, name, pArray);
    if (status != 0)
    {
        printf("\n>>>>> Failed to put mxArray \"%s\" into MAT-file.\n", name);
        cout << flush;
        MPI_Abort(MPI_COMM_WORLD, -1);
    }

    // Free memory allocated for pArray, but do not free memory allocated for vector.data
    mxSetData(pArray, nullptr);
    mxDestroyArray(pArray);
}

template <typename T>
void WriteVector(const LocalVector<T> &vector, const std::string &name, MatFile matFile)
{
    WriteVector(vector, name.c_str(), matFile);
}

// Gather and write a distributed vector of special type (e.g. "std::mt19937", "mod::hh_e<float>" etc.) to "intermediate.mat".
// The method is called on all ranks with any number of threads, but not inside "#pragma omp master" region.
template <typename T>
void GatherWriteSpecialVectorToIntermediateMat(const DistVector<T> &vector, const char *name)
{
    using namespace DistEnv;
    
    // Allocate memory on master thread of master rank
    int length = 0;
    #pragma omp master
    {
        length = vector.length;
    }
    LocalVector<T> localVector1(length, AllocMode::onlyMasterMxCalloc);

    // Gather the vector
    vector.Gather(localVector1, true);

    #pragma omp master
    {
        if (myRank == MASTER_RANK)
        {
            // Little hack to make it possible to serialize objects of type "Mersenne twister engine" or "MOD current" to the MAT-file
            // !! TODO: Investigate why Linux compiler does not allow commented code while MSVS compiler allows
            /*
            LocalVector<uint8_t> localVector2 = localVector1.CastReshape<uint8_t>();
            */
            uint8_t *typeMarker = nullptr;
            LocalVector<uint8_t> localVector2 = localVector1.CastReshape(typeMarker);
            WriteVector<uint8_t>(localVector2, name, MatFile::Intermediate);

            // Deallocate memory
            mxFree(localVector1.data);
        }
    }
}

template <typename T, typename RNGT>
void WriteMatrix(const LocalDenseMatrix<T, RNGT> &matrix, const char *name, MatFile matFile)
{
    using namespace pMat;

    // Determine whether file "output.mat" or "intermediate.mat" or should be used to write to
    MATFile *pFile = nullptr;
    switch (matFile)
    {
    case MatFile::Output:
        {
            pFile = pOutMatFile;
            break;
        }
    case MatFile::Intermediate:
        {
            pFile = pIntermMatFile;
            break;
        }
    default:
        {
            cout << "\n>>>>> Bad matFile argument passed to WriteMatrix method.\n" << flush;
            MPI_Abort(MPI_COMM_WORLD, -1);
        }
    }

    mxClassID classId = GetMxClassId<T>();

    mxArray *pArray = mxCreateNumericMatrix(matrix.numRows, matrix.numCols, classId, mxREAL);

    mxSetData(pArray, matrix.data);

    int status = matPutVariable(pFile, name, pArray);
    if (status != 0)
    {
        printf("\n>>>>> Failed to put mxArray \"%s\" into MAT-file.\n", name);
        cout << flush;
        MPI_Abort(MPI_COMM_WORLD, -1);
    }

    // Free memory allocated for pArray, but do not free memory allocated for vector.data
    mxSetData(pArray, nullptr);
    mxDestroyArray(pArray);
}

// A wrapper of "WriteVector" which cuts the vector to avoid saving uninitialized elements to MAT-file
template <typename T>
void WriteCutVector(LocalVector<T> &vector, const char *name, int currentLength, MatFile matFile)
{
    int totalLength = vector.length;
    vector.length = currentLength;
    WriteVector<T>(vector, name, matFile);
    vector.length = totalLength;
}

// A wrapper of "WriteMatrix" which cuts the matrix to avoid saving uninitialized columns to MAT-file
template <typename T, typename RNGT>
void WriteCutMatrix(LocalDenseMatrix<T, RNGT> &matrix, const char *name, int currentNumCols, MatFile matFile)
{
    int totalNumCols = matrix.numCols;
    matrix.numCols = currentNumCols;
    WriteMatrix<T, RNGT>(matrix, name, matFile);
    matrix.numCols = totalNumCols;
}

template <typename T, typename RNGT>
void WriteCutMatrix(LocalDenseMatrix<T, RNGT> &matrix, const std::string &name, int currentNumCols, MatFile matFile)
{
    WriteCutMatrix(matrix, name.c_str(), currentNumCols, matFile);
}

// Given name, write mxArray to the output MAT-file.
// The method is called just on master thread of master rank.
void WriteMxArray(mxArray *pArray, const char *name)
{
    using namespace pMat;

    MATFile *pDestFile = pOutMatFile;

    int status = matPutVariable(pDestFile, name, pArray);

    if (status != 0)
    {
        printf("\n>>>>> Failed to put mxArray \"%s\" into output MAT-file.\n", name);
        cout << flush;
        MPI_Abort(MPI_COMM_WORLD, -1);
    }
}

void WriteMxArray(mxArray *pArray, const std::string &name)
{
    WriteMxArray(pArray, name.c_str());
}

#pragma endregion

#pragma region Writer instantiations

// WriteScalar

template
void WriteScalar<float>(float scalar, const char *name, MatFile matFile);

template
void WriteScalar<double>(double scalar, const char *name, MatFile matFile);

template
void WriteScalar<int>(int scalar, const char *name, MatFile matFile);

template
void WriteScalar<bool>(bool scalar, const char *name, MatFile matFile);

template
void WriteScalar<int>(int scalar, const std::string &name, MatFile matFile);

// WriteVector

template
void WriteVector<float>(const LocalVector<float> &vector, const char *name, MatFile matFile);

template
void WriteVector<double>(const LocalVector<double> &vector, const char *name, MatFile matFile);

template
void WriteVector<int>(const LocalVector<int> &vector, const char *name, MatFile matFile);

template
void WriteVector<uint8_t>(const LocalVector<uint8_t> &vector, const char *name, MatFile matFile);

template
void WriteVector<float>(const LocalVector<float> &vector, const std::string &name, MatFile matFile);

template
void WriteVector<double>(const LocalVector<double> &vector, const std::string &name, MatFile matFile);

// GatherWriteSpecialVectorToIntermediateMat
template
void GatherWriteSpecialVectorToIntermediateMat<std::mt19937>(const DistVector<std::mt19937> &vector, const char *name);

template
void GatherWriteSpecialVectorToIntermediateMat<std::mt19937_64>(const DistVector<std::mt19937_64> &vector, const char *name);

// Instantiations of the method for MOD classes
#include "ModCurrents/Autogenerated/MatFileWriteUtilsIncl.cpp"

// WriteMatrix

template
void WriteMatrix<float, std::mt19937>(const LocalDenseMatrix<float, std::mt19937> &matrix, const char *name, MatFile matFile);

template
void WriteMatrix<double, std::mt19937>(const LocalDenseMatrix<double, std::mt19937> &matrix, const char *name, MatFile matFile);

template
void WriteMatrix<double, std::mt19937_64>(const LocalDenseMatrix<double, std::mt19937_64> &matrix, const char *name, MatFile matFile);

template
void WriteMatrix<bool, std::mt19937>(const LocalDenseMatrix<bool, std::mt19937> &matrix, const char *name, MatFile matFile);

template
void WriteMatrix<bool, std::mt19937_64>(const LocalDenseMatrix<bool, std::mt19937_64> &matrix, const char *name, MatFile matFile);

template
void WriteMatrix<int, std::mt19937>(const LocalDenseMatrix<int, std::mt19937> &matrix, const char *name, MatFile matFile);

template
void WriteMatrix<int, std::mt19937_64>(const LocalDenseMatrix<int, std::mt19937_64> &matrix, const char *name, MatFile matFile);

// WriteCutVector

template
void WriteCutVector<float>(LocalVector<float> &vector, const char *name, int currentLength, MatFile matFile);

template
void WriteCutVector<double>(LocalVector<double> &vector, const char *name, int currentLength, MatFile matFile);

template
void WriteCutVector<int>(LocalVector<int> &vector, const char *name, int currentLength, MatFile matFile);

// WriteCutMatrix

template
void WriteCutMatrix<float, std::mt19937>(LocalDenseMatrix<float, std::mt19937> &matrix, const char *name, int currentNumCols, MatFile matFile);

template
void WriteCutMatrix<double, std::mt19937>(LocalDenseMatrix<double, std::mt19937> &matrix, const char *name, int currentNumCols, MatFile matFile);

template
void WriteCutMatrix<double, std::mt19937_64>(LocalDenseMatrix<double, std::mt19937_64> &matrix, const char *name, int currentNumCols, MatFile matFile);

template
void WriteCutMatrix<float, std::mt19937>(LocalDenseMatrix<float, std::mt19937> &matrix, const std::string &name, int currentNumCols, MatFile matFile);

template
void WriteCutMatrix<double, std::mt19937>(LocalDenseMatrix<double, std::mt19937> &matrix, const std::string &name, int currentNumCols, MatFile matFile);

template
void WriteCutMatrix<double, std::mt19937_64>(LocalDenseMatrix<double, std::mt19937_64> &matrix, const std::string &name, int currentNumCols, MatFile matFile);

#pragma endregion