function TranslateModFilesAndDeployAutogenFiles(md5FileName)
%% Translate all the MOD files to C++ source and header files for both "e" and "i" neurons,
%  create an empty MD5-file,
%  then deploy the files to the worker source code directory
    
    global importMod_e modDirPath_e modFileNameToUninitParamsNamesStruct_e
    global importMod_i modDirPath_i modFileNameToUninitParamsNamesStruct_i
    global HostDirTypes
    
    % Create a new or clean up the existing temporary directory
    tempDirName = '_ModTranslatorTemp';
    outDirPath = fullfile(GetDirPath(HostDirTypes.Temp), tempDirName);
    PrepareEmptyDir(outDirPath);
    
    disp('Translating MOD files into C++ code ...');
    
    % Generate C++ files (H and CPP files pairs) for all the selected MOD files
    cppFileNames_e = TranslateModFilesToCpp(importMod_e, modDirPath_e, modFileNameToUninitParamsNamesStruct_e, outDirPath, 'e');
    cppFileNames_i = TranslateModFilesToCpp(importMod_i, modDirPath_i, modFileNameToUninitParamsNamesStruct_i, outDirPath, 'i');

    disp('Preparing intermediary C++ code ...');
    
    % Generate cumulative *.H and *.CPP files
    GenerateCumulativeHdrFile(cppFileNames_e, cppFileNames_i, outDirPath);
    GenerateCumulativeSrcFile(cppFileNames_e, cppFileNames_i, outDirPath);

    % Generate *Incl.* files
    GenerateInclFiles(cppFileNames_e, cppFileNames_i, outDirPath);

    % Create an empty MD5-file
    md5FilePathName = fullfile(outDirPath, md5FileName);
    fid = fopen(md5FilePathName, 'w');
    assert(fid ~= -1);
    fclose(fid);
    
    % Deploy all the generated C++ files and MD5-file
    success = DeployAutogenFiles(tempDirName);
    
    % Remove the temporary directory
    rmdir(outDirPath, 's');
    
    assert(success, 'Failed to deploy the autogenerated files.');
    
end

function cppFileNames = TranslateModFilesToCpp(importMod, modDirPath, modFileNameToUninitParamsNamesStruct, outDirPath, neuronType)
%% Translate all the MOD files to C++ source and header files for one neuron type ("e" or "i")

    if isempty(importMod) || ~importMod
        cppFileNames = {};
        return
    end
    
    modFilesNames = fields(modFileNameToUninitParamsNamesStruct);
    
    numModFiles = length(modFilesNames);
    paths_to_mod = cell(numModFiles, 1);
    for i = 1 : numModFiles
        paths_to_mod{i} = fullfile(modDirPath, [modFilesNames{i}, '.mod']);
    end

    cppFileNames = cell(length(paths_to_mod), 1);
    arr_pars = mod_params_parser(neuronType);
    for i = 1 : length(paths_to_mod)
        path_to_mod = paths_to_mod{i};
        [~, modFileName, ~] = fileparts(path_to_mod);
        
        userInitParams = struct;
        for j = 1 : length(arr_pars)
            if strcmp(arr_pars{j}.filename, modFileName)
                userInitParams = arr_pars{j};
                break
            end
        end
             
        cppFileName = [modFileName, '_', neuronType];
        cppFileNames{i, 1} = cppFileName;
        
        fprintf('    Parsing %s.mod ...\n', modFileName);
        [parametersBlocks, freeLocalVars] = ParseModFile(path_to_mod);
        
        fprintf('    Generating %s.h ...\n', cppFileName);
        GenerateHdrFile(cppFileName, parametersBlocks, freeLocalVars, outDirPath, userInitParams);
    
        fprintf('    Generating %s.cpp ...\n', cppFileName);
        GenerateSrcFile(cppFileName, parametersBlocks, outDirPath);

    end
    
end
