function GenerateCumulativeHdrFile(modClassNames_e, modClassNames_i, outDirPath)
%%
    outFileName = 'AllModCurrents.h';
    
    fprintf('    Generating %s ...\n', outFileName);
    
    numModClasses_e = length(modClassNames_e);
    numModClasses_i = length(modClassNames_i);
    
    lines = {'#pragma once'};
    lines{end + 1, 1} = '';
    lines{end + 1} = '#include <vector>';
    lines{end + 1} = '#include <memory>';
    lines{end + 1} = '#include <tuple>';
    lines{end + 1} = '';
    lines{end + 1} = '#include "Containers/DistVector.h"';
    lines{end + 1} = '#include "MatFileIO/MatFileIOUtils.h"';
    lines{end + 1} = '#include "../AllModCurrentsBase.h"';
    lines{end + 1} = '';
 
    pat =            '#include "%s.h"';
    for i = 1 : numModClasses_e
        lines{end + 1} = sprintf(pat, modClassNames_e{i}); %#ok<*AGROW>
    end
    lines{end + 1} = '';
    for i = 1 : numModClasses_i
        lines{end + 1} = sprintf(pat, modClassNames_i{i});
    end
    
    lines{end + 1} = '';
    lines = GenerateChildCppClass(lines, modClassNames_e, 'e');
    lines{end + 1} = '';
    lines = GenerateChildCppClass(lines, modClassNames_i, 'i');

    SaveLinesToFile(lines, outDirPath, outFileName);
    
end

function lines = GenerateChildCppClass(lines, modClassNames, neuronType)
%%
    lines{end + 1} = 'template <typename T>';
    lines{end + 1} = ['class AllModCurrents_', neuronType, ' : public AllModCurrentsBase<T>'];
    
    numModClasses = length(modClassNames);

    if numModClasses == 0
        lines{end + 1} = '{';
        lines{end + 1} = 'public:';
        lines{end + 1} = ['    AllModCurrents_', neuronType, '(int num_', neuronType, ', bool continuationMode)'];
        lines{end + 1} = ['    : AllModCurrentsBase<T>(num_', neuronType, ')'];
        lines{end + 1} = '    { }';
        lines{end + 1} = '};';
        return
    end

    lines{end + 1} = '{';
    
    % Generate private data
    lines{end + 1} = 'private:';
    lines{end + 1} = '// The only data member of this class';
    lines{end + 1} = 'std::tuple<';
    pat =            '    DistVector<mod::%s<T>>%s';
    for i = 1 : numModClasses
        if i < numModClasses
            lineEnd = ',';
        else
            lineEnd = '> distCurrentsTuple;';
        end
        lines{end + 1} = sprintf(pat, modClassNames{i}, lineEnd);
    end
    
    lines{end + 1} = '';
    
    % Generate public methods
    
    lines{end + 1} = 'public:';
    lines{end + 1} = '';
    
    % Generate custom constructor
    lines = GenerateCustomCtor(lines, modClassNames, neuronType);

    % Generate SetVoltage method
    lines = GenerateSetVoltage(lines, numModClasses);
    
    % Generate GetSumCurrent method
    lines = GenerateGetSumCurrent(lines, numModClasses);
    
    % Generate DoOneStepPart1 method
    lines = GenerateDoOneStepPart1(lines, numModClasses);
    
    % Generate DoOneStepPart2 method
    lines = GenerateDoOneStepPart2(lines, numModClasses);
    
    % Generate GatherWriteIntermediateData method
    lines = GenerateGatherWriteIntermediateData(lines, modClassNames);
    
    lines{end + 1} = '';
    lines{end + 1} = '};';
    
end

function lines = GenerateCustomCtor(lines, modClassNames, neuronType)
%% Generate custom constructor

    numModClasses = length(modClassNames);

    lines{end + 1} = '// This constructor is called on all ranks, but in master thread only';
    lines{end + 1} = ['AllModCurrents_', neuronType, '(int num_', neuronType, ', bool continuationMode)'];
    lines{end + 1} = [': AllModCurrentsBase<T>(num_', neuronType, ')'];
    lines{end + 1} = '{';
    lines = GenerateMakeTupleCall(lines, modClassNames, neuronType);
    lines{end + 1} = '    ';
    pat =            '    auto &distCurrents_%i = std::get<%i>(distCurrentsTuple);';
    for i = 0 : numModClasses - 1
        lines{end + 1} = sprintf(pat, i, i);
    end
    lines{end + 1} = '    ';
    lines{end + 1} = '    int localLength = this->I.localLength;';
    lines{end + 1} = '    if (continuationMode)';
    lines{end + 1} = '    {';
    pat =            '        auto distCurrentsOld_%i = ReadCheckScatterSpecialVectorFromIntermediateMat<mod::%s<T>>("%s", num_%s);';
    for i = 1 : numModClasses
        modClassName = modClassNames{i};
        matVarName = ModClassNameToMatVarName(modClassName);
        lines{end + 1} = sprintf(pat, i - 1, modClassName, matVarName, neuronType);
    end
    lines{end + 1} = '        ';
    lines{end + 1} = '        for (int idx = 0; idx < localLength; idx++)';
    lines{end + 1} = '        {';
    pat1 =           '            distCurrents_%i[idx] = distCurrentsOld_%i[idx];';
    pat2 =           '            distCurrents_%i[idx].initBasePtrs();';
    for i = 0 : numModClasses - 1
        lines{end + 1} = sprintf(pat1, i, i);
        lines{end + 1} = sprintf(pat2, i);
    end
    lines{end + 1} = '        }';
    lines{end + 1} = '        ';
    pat =            '        delete[] distCurrentsOld_%i.localData;';
    for i = 0 : numModClasses - 1
        lines{end + 1} = sprintf(pat, i);
    end
    lines{end + 1} = '    }';
    lines{end + 1} = '    else';
    lines{end + 1} = '    {';
    lines{end + 1} = '        for (int idx = 0; idx < localLength; idx++)';
    lines{end + 1} = '        {';
    pat =            '            distCurrents_%i[idx].init();';
    for i = 0 : numModClasses - 1
        lines{end + 1} = sprintf(pat, i);
    end
    lines{end + 1} = '        }';
    lines{end + 1} = '    }';
    lines{end + 1} = '}';
    lines{end + 1} = '';
    
end

function lines = GenerateMakeTupleCall(lines, modClassNames, neuronType)
%%
    numModClasses = length(modClassNames);

    lines{end + 1} = '    distCurrentsTuple = std::make_tuple(';
    if numModClasses ~= 0
        pat = '        DistVector<mod::%s<T>>(num_%s)%s';
        for i = 1 : numModClasses
            modClassName = modClassNames{i};
            if i < numModClasses
                lineEnd = ',';
            else
                lineEnd = ');';
            end
            lines{end + 1} = sprintf(pat, modClassName, neuronType, lineEnd);
        end
    else
        lines{end} = [lines{end}, ');'];
    end
    
end

function lines = GenerateSetVoltage(lines, numModClasses)
%% Generate SetVoltage method

    lines{end + 1} = '// This method is called on all ranks with all threads';
    lines{end + 1} = 'void SetVoltage(const DistVector<T> &v) override';
    lines = GenerateCommonBlock(lines, numModClasses);
    pat =            '        atomicCurrent_%i.v = v[idx];';
    for i = 0 : numModClasses - 1
        lines{end + 1} = sprintf(pat, i);
    end
    lines{end + 1} = '    }';
    lines{end + 1} = '}';
    lines{end + 1} = '';
    
end
    
function lines = GenerateGetSumCurrent(lines, numModClasses)
%% Generate GetSumCurrent method

    lines{end + 1} = '// This method is called on all ranks with all threads';
    lines{end + 1} = 'DistVector<T> GetSumCurrent() override';
    lines = GenerateCommonBlock(lines, numModClasses);
    lines{end + 1} = '        T I_sum = 0;';
    pat =            '        I_sum += atomicCurrent_%i.getResCurrent();';
    for i = 0 : numModClasses - 1
        lines{end + 1} = sprintf(pat, i);
    end
    lines{end + 1} = '        this->I[idx] = I_sum;';
    lines{end + 1} = '    }';
    lines{end + 1} = '    ';
    lines{end + 1} = '    return this->I;';
    lines{end + 1} = '}';
    lines{end + 1} = '';
    
end

function lines = GenerateDoOneStepPart1(lines, numModClasses)
%% Generate DoOneStepPart1 method

    lines{end + 1} = '// This method is called on all ranks with all threads';
    lines{end + 1} = 'void DoOneStepPart1(const DistVector<T> &v, T dt05) override';
    lines = GenerateCommonBlock(lines, numModClasses);
    for i = 0 : numModClasses - 1
        lines{end + 1} = sprintf('        atomicCurrent_%i.v = v[idx];', i);
        lines{end + 1} = sprintf('        atomicCurrent_%i.states();', i);
        lines{end + 1} = sprintf('        atomicCurrent_%i.integrate(dt05);', i);
        lines{end + 1} = sprintf('        atomicCurrent_%i.currents();', i);
        lines{end + 1} = '        ';
    end
    lines = GenerateDoOneStepCommonEndBlock(lines, numModClasses, 'dt05');
    
end

function lines = GenerateDoOneStepPart2(lines, numModClasses)
%% Generate DoOneStepPart2 method

    lines{end + 1} = '// This method is called on all ranks with all threads';
    lines{end + 1} = 'void DoOneStepPart2(const DistVector<T> &v, const DistVector<T> &v_tmp, T dt) override';
    lines = GenerateCommonBlock(lines, numModClasses);
    for i = 0 : numModClasses - 1
        lines{end + 1} = sprintf('        atomicCurrent_%i.v = v_tmp[idx];', i);
        lines{end + 1} = sprintf('        atomicCurrent_%i.states();', i);
        lines{end + 1} = sprintf('        atomicCurrent_%i.integrate(dt);', i);
        lines{end + 1} = sprintf('        atomicCurrent_%i.v = v[idx];', i);
        lines{end + 1} = sprintf('        atomicCurrent_%i.currents();', i);
        lines{end + 1} = '        ';
    end
    lines = GenerateDoOneStepCommonEndBlock(lines, numModClasses, 'dt / 2');
    
end

function lines = GenerateGatherWriteIntermediateData(lines, modClassNames)
%% Generate GatherWriteIntermediateData method

    lines{end + 1} = '// This method is called on all ranks, but in master thread only';
    lines{end + 1} = 'void GatherWriteIntermediateData() override';
    lines{end + 1} = '{';
    pat =            '    GatherWriteSpecialVectorToIntermediateMat(std::get<%i>(distCurrentsTuple), "%s");';
    for i = 1 : length(modClassNames)
        matVarName = ModClassNameToMatVarName(modClassNames{i});
        lines{end + 1} = sprintf(pat, i - 1, matVarName);
    end
    lines{end + 1} = '}';
    
end

function matVarName = ModClassNameToMatVarName(modClassName)
%%
    matVarName = ['modCurState_', modClassName];
end

function lines = GenerateCommonBlock(lines, numModClasses)
%% Generate a common block of code used in the next methods:
%  SetVoltage, GetSumCurrent, DoOneStepPart1, DoOneStepPart2

    lines{end + 1} = '{';
    lines{end + 1} = '    using namespace DistEnv;';
    lines{end + 1} = '    ';
    lines{end + 1} = '    int startIdx, endIdx;';
    lines{end + 1} = '    GetMyThreadChunkStartEndIdxs(this->I.localLength, startIdx, endIdx);';
    lines{end + 1} = '    ';
    pat =            '    auto &distCurrents_%i = std::get<%i>(distCurrentsTuple);';
    for i = 0 : numModClasses - 1
        lines{end + 1} = sprintf(pat, i, i);
    end
    lines{end + 1} = '    ';
    lines{end + 1} = '    for (int idx = startIdx; idx < endIdx; idx++)';
    lines{end + 1} = '    {';
    pat =            '        auto &atomicCurrent_%i = distCurrents_%i[idx];';
    for i = 0 : numModClasses - 1
        lines{end + 1} = sprintf(pat, i, i);
    end
    lines{end + 1} = '        ';
    
end

function lines = GenerateDoOneStepCommonEndBlock(lines, numModClasses, dtInc)
%% Generate a common block of code used in the end of the next methods:
%  DoOneStepPart1, DoOneStepPart2

    lines{end + 1} = '        T I_sum = 0;';
    pat =            '        I_sum += atomicCurrent_%i.getResCurrent();';
    for i = 0 : numModClasses - 1
        lines{end + 1} = sprintf(pat, i);
    end
    lines{end + 1} = '        this->I[idx] = I_sum;';
    lines{end + 1} = '    }';
    lines{end + 1} = '    ';
    lines{end + 1} = '    #pragma omp barrier';
    lines{end + 1} = '    #pragma omp master';
    lines{end + 1} = '    {';
    lines{end + 1} = sprintf('        AtomicModCurrentBase<T>::t += %s;', dtInc);  
    lines{end + 1} = '    }';
    lines{end + 1} = '    #pragma omp barrier';
    lines{end + 1} = '}';
    lines{end + 1} = '';
    
end